<?php

namespace ShareFileConnector\Tests;

use PHPUnit\Framework\TestCase;
use ShareFileConnector\ShareFileAPI;
use ShareFileConnector\ClientManager;

class ShareFileAPITest extends TestCase
{
    private ShareFileAPI $shareFileAPI;
    private ClientManager $clientManager;

    protected function setUp(): void
    {
        // Load environment variables for testing
        $dotenv = \Dotenv\Dotenv::createImmutable(__DIR__ . '/..');
        $dotenv->load();

        $this->shareFileAPI = new ShareFileAPI();
        $this->clientManager = new ClientManager($this->shareFileAPI);
    }

    /**
     * Test client creation with valid data
     */
    public function testCreateClientWithValidData(): void
    {
        $companyData = [
            'coID' => 'TEST123',
            'coName' => 'Test Company',
            'firstName' => 'John',
            'lastName' => 'Doe',
            'email' => 'john.doe@testcompany.com'
        ];

        $result = $this->clientManager->processNewCompany($companyData);

        $this->assertIsArray($result);
        $this->assertArrayHasKey('success', $result);
        
        if ($result['success']) {
            $this->assertArrayHasKey('clientId', $result);
            $this->assertNotEmpty($result['clientId']);
        }
    }

    /**
     * Test client creation with missing required fields
     */
    public function testCreateClientWithMissingFields(): void
    {
        $companyData = [
            'coID' => 'TEST124',
            'coName' => 'Test Company',
            // Missing firstName, lastName, email
        ];

        $result = $this->clientManager->processNewCompany($companyData);

        $this->assertIsArray($result);
        $this->assertFalse($result['success']);
        $this->assertArrayHasKey('message', $result);
        $this->assertStringContainsString('Missing required field', $result['message']);
    }

    /**
     * Test client creation with invalid email
     */
    public function testCreateClientWithInvalidEmail(): void
    {
        $companyData = [
            'coID' => 'TEST125',
            'coName' => 'Test Company',
            'firstName' => 'John',
            'lastName' => 'Doe',
            'email' => 'invalid-email'
        ];

        $result = $this->clientManager->processNewCompany($companyData);

        $this->assertIsArray($result);
        $this->assertFalse($result['success']);
        $this->assertArrayHasKey('message', $result);
        $this->assertStringContainsString('Invalid email format', $result['message']);
    }

    /**
     * Test duplicate client creation
     */
    public function testDuplicateClientCreation(): void
    {
        $companyData = [
            'coID' => 'TEST126',
            'coName' => 'Test Company',
            'firstName' => 'John',
            'lastName' => 'Doe',
            'email' => 'john.doe@testcompany.com'
        ];

        // Create client first time
        $result1 = $this->clientManager->processNewCompany($companyData);
        
        // Try to create same client again
        $result2 = $this->clientManager->processNewCompany($companyData);

        $this->assertIsArray($result2);
        $this->assertFalse($result2['success']);
        $this->assertArrayHasKey('message', $result2);
        $this->assertStringContainsString('already exists', $result2['message']);
    }

    /**
     * Test client update functionality
     */
    public function testUpdateClient(): void
    {
        $companyData = [
            'coID' => 'TEST127',
            'coName' => 'Test Company',
            'firstName' => 'John',
            'lastName' => 'Doe',
            'email' => 'john.doe@testcompany.com'
        ];

        // Create client first
        $createResult = $this->clientManager->processNewCompany($companyData);
        
        if ($createResult['success']) {
            // Update client
            $updateData = [
                'coID' => 'TEST127',
                'coName' => 'Updated Test Company',
                'firstName' => 'Jane',
                'lastName' => 'Smith',
                'email' => 'jane.smith@testcompany.com'
            ];

            $updateResult = $this->clientManager->updateCompany($updateData);

            $this->assertIsArray($updateResult);
            $this->assertTrue($updateResult['success']);
            $this->assertArrayHasKey('clientId', $updateResult);
        }
    }

    /**
     * Test getting client by ClientID
     */
    public function testGetClientByClientId(): void
    {
        $clientId = 'TEST128';
        $companyData = [
            'coID' => $clientId,
            'coName' => 'Test Company',
            'firstName' => 'John',
            'lastName' => 'Doe',
            'email' => 'john.doe@testcompany.com'
        ];

        // Create client first
        $this->clientManager->processNewCompany($companyData);

        // Get client by ClientID
        $client = $this->shareFileAPI->getClientByClientId($clientId);

        $this->assertIsArray($client);
        $this->assertArrayHasKey('Id', $client);
        $this->assertArrayHasKey('ClientID', $client);
        $this->assertEquals($clientId, $client['ClientID']);
    }
} 