<?php

namespace ShareFileConnector;

class SimpleShareFileAPI
{
    private string $accessToken;
    private string $subdomain;
    private string $baseUrl;
    private string $logFile;

    public function __construct()
    {
        $this->subdomain = $_ENV['SHAREFILE_SUBDOMAIN'] ?? '';
        $this->baseUrl = "https://{$this->subdomain}.sf-api.com/sf/v3";
        $this->logFile = $_ENV['LOG_FILE'] ?? 'logs/sharefile_connector.log';
        
        // Create logs directory if it doesn't exist
        $logDir = dirname($this->logFile);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        $this->authenticate();
    }

    /**
     * Log messages to file
     */
    private function log(string $level, string $message): void
    {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[{$timestamp}] [{$level}] {$message}" . PHP_EOL;
        file_put_contents($this->logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }

    /**
     * Make HTTP request
     */
    private function makeRequest(string $url, string $method = 'GET', array $data = null, array $headers = []): array
    {
        $ch = curl_init();
        
        $defaultHeaders = [
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        if ($this->accessToken) {
            $defaultHeaders[] = 'Authorization: Bearer ' . $this->accessToken;
        }
        
        $headers = array_merge($defaultHeaders, $headers);
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_FOLLOWLOCATION => true
        ]);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        } elseif ($method === 'PATCH') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PATCH');
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            throw new \Exception("cURL error: {$error}");
        }
        
        return [
            'status_code' => $httpCode,
            'body' => $response
        ];
    }

    /**
     * Authenticate with ShareFile using OAuth2
     */
    private function authenticate(): void
    {
        try {
            $url = "https://{$this->subdomain}.sharefile.com/oauth/token";
            $data = [
                'grant_type' => 'password',
                'client_id' => $_ENV['SHAREFILE_CLIENT_ID'],
                'client_secret' => $_ENV['SHAREFILE_CLIENT_SECRET'],
                'username' => $_ENV['SHAREFILE_USERNAME'],
                'password' => $_ENV['SHAREFILE_PASSWORD']
            ];
            
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => http_build_query($data),
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/x-www-form-urlencoded'
                ],
                CURLOPT_TIMEOUT => 30,
                CURLOPT_SSL_VERIFYPEER => true
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($error) {
                throw new \Exception("cURL error: {$error}");
            }
            
            if ($httpCode !== 200) {
                throw new \Exception("Authentication failed with HTTP code: {$httpCode}");
            }
            
            $data = json_decode($response, true);
            if (!isset($data['access_token'])) {
                throw new \Exception("No access token in response");
            }
            
            $this->accessToken = $data['access_token'];
            $this->log('INFO', 'Successfully authenticated with ShareFile API');
            
        } catch (\Exception $e) {
            $this->log('ERROR', 'Authentication failed: ' . $e->getMessage());
            throw new \Exception('Failed to authenticate with ShareFile API: ' . $e->getMessage());
        }
    }

    /**
     * Create a new client in ShareFile
     */
    public function createClient(array $clientData): array
    {
        try {
            $payload = [
                'AccountType' => 'Company',
                'CompanyName' => $clientData['companyName'],
                'FirstName' => $clientData['firstName'],
                'LastName' => $clientData['lastName'],
                'Email' => $clientData['email'],
                'ClientID' => $clientData['clientId'],
                'InternalDisplayName' => $clientData['internalDisplayName']
            ];

            $response = $this->makeRequest($this->baseUrl . '/Accounts', 'POST', $payload);
            
            if ($response['status_code'] !== 201 && $response['status_code'] !== 200) {
                throw new \Exception("API returned status code: {$response['status_code']}");
            }

            $result = json_decode($response['body'], true);
            $this->log('INFO', "Successfully created ShareFile client: {$clientData['companyName']} (ID: {$result['Id']})");
            
            return $result;
        } catch (\Exception $e) {
            $this->log('ERROR', "Failed to create ShareFile client {$clientData['companyName']}: " . $e->getMessage());
            throw new \Exception("Failed to create ShareFile client: " . $e->getMessage());
        }
    }

    /**
     * Create a folder from template for a client
     */
    public function createFolderFromTemplate(string $clientId, string $folderName, string $templateFolderId): array
    {
        try {
            $payload = [
                'Name' => $folderName,
                'TemplateId' => $templateFolderId,
                'ParentId' => $clientId
            ];

            $response = $this->makeRequest($this->baseUrl . '/Items', 'POST', $payload);
            
            if ($response['status_code'] !== 201 && $response['status_code'] !== 200) {
                throw new \Exception("API returned status code: {$response['status_code']}");
            }

            $result = json_decode($response['body'], true);
            $this->log('INFO', "Successfully created folder from template: {$folderName} (ID: {$result['Id']})");
            
            return $result;
        } catch (\Exception $e) {
            $this->log('ERROR', "Failed to create folder from template {$folderName}: " . $e->getMessage());
            throw new \Exception("Failed to create folder from template: " . $e->getMessage());
        }
    }

    /**
     * Get client information by ClientID
     */
    public function getClientByClientId(string $clientId): ?array
    {
        try {
            $response = $this->makeRequest($this->baseUrl . "/Accounts?ClientID={$clientId}");
            
            if ($response['status_code'] !== 200) {
                $this->log('ERROR', "Failed to get client by ClientID {$clientId}: HTTP {$response['status_code']}");
                return null;
            }

            $data = json_decode($response['body'], true);
            
            if (!empty($data['value'])) {
                return $data['value'][0];
            }
            
            return null;
        } catch (\Exception $e) {
            $this->log('ERROR', "Failed to get client by ClientID {$clientId}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Update client information
     */
    public function updateClient(string $clientId, array $updateData): array
    {
        try {
            $response = $this->makeRequest($this->baseUrl . "/Accounts/{$clientId}", 'PATCH', $updateData);
            
            if ($response['status_code'] !== 200) {
                throw new \Exception("API returned status code: {$response['status_code']}");
            }

            $result = json_decode($response['body'], true);
            $this->log('INFO', "Successfully updated ShareFile client: {$clientId}");
            
            return $result;
        } catch (\Exception $e) {
            $this->log('ERROR', "Failed to update ShareFile client {$clientId}: " . $e->getMessage());
            throw new \Exception("Failed to update ShareFile client: " . $e->getMessage());
        }
    }
} 