<?php

namespace ShareFileConnector;

class ClientManager
{
    private $shareFileAPI;
    private $logFile;

    public function __construct($shareFileAPI)
    {
        $this->shareFileAPI = $shareFileAPI;
        $this->logFile = $_ENV['LOG_FILE'] ?? 'logs/sharefile_connector.log';
        
        // Create logs directory if it doesn't exist
        $logDir = dirname($this->logFile);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
    }

    /**
     * Log messages to file
     */
    private function log(string $level, string $message): void
    {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[{$timestamp}] [{$level}] {$message}" . PHP_EOL;
        file_put_contents($this->logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }

    /**
     * Process a new company creation from eProject
     */
    public function processNewCompany(array $companyData): array
    {
        try {
            $this->log('INFO', "Processing new company: {$companyData['coName']} (coID: {$companyData['coID']})");

            // Validate required data
            $this->validateCompanyData($companyData);

            // Check if client already exists in ShareFile
            $existingClient = $this->shareFileAPI->getClientByClientId($companyData['coID']);
            if ($existingClient) {
                $this->log('WARNING', "Client already exists in ShareFile: {$companyData['coName']} (ClientID: {$companyData['coID']})");
                return [
                    'success' => false,
                    'message' => 'Client already exists in ShareFile',
                    'clientId' => $existingClient['Id']
                ];
            }

            // Prepare client data for ShareFile
            $clientData = $this->prepareClientData($companyData);

            // Create client in ShareFile
            $shareFileClient = $this->shareFileAPI->createClient($clientData);

            // Create folder from template if template ID is configured
            $folderResult = null;
            if (!empty($_ENV['SHAREFILE_TEMPLATE_FOLDER_ID'])) {
                $folderResult = $this->createProjectFolder($shareFileClient['Id'], $companyData);
            }

            $this->log('INFO', "Successfully processed new company: {$companyData['coName']}");

            return [
                'success' => true,
                'message' => 'Company successfully created in ShareFile',
                'clientId' => $shareFileClient['Id'],
                'folderId' => $folderResult['Id'] ?? null
            ];

        } catch (\Exception $e) {
            $this->log('ERROR', "Failed to process new company {$companyData['coName']}: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to process company: ' . $e->getMessage(),
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Update existing company information in ShareFile
     */
    public function updateCompany(array $companyData): array
    {
        try {
            $this->log('INFO', "Updating company: {$companyData['coName']} (coID: {$companyData['coID']})");

            // Validate required data
            $this->validateCompanyData($companyData);

            // Find existing client in ShareFile
            $existingClient = $this->shareFileAPI->getClientByClientId($companyData['coID']);
            if (!$existingClient) {
                $this->log('WARNING', "Client not found in ShareFile: {$companyData['coName']} (ClientID: {$companyData['coID']})");
                return [
                    'success' => false,
                    'message' => 'Client not found in ShareFile'
                ];
            }

            // Prepare update data
            $updateData = [
                'CompanyName' => $companyData['coName'],
                'FirstName' => $companyData['firstName'],
                'LastName' => $companyData['lastName'],
                'Email' => $companyData['email'],
                'InternalDisplayName' => $companyData['coName'] . ' (' . $companyData['coID'] . ')'
            ];

            // Update client in ShareFile
            $updatedClient = $this->shareFileAPI->updateClient($existingClient['Id'], $updateData);

            $this->log('INFO', "Successfully updated company: {$companyData['coName']}");

            return [
                'success' => true,
                'message' => 'Company successfully updated in ShareFile',
                'clientId' => $updatedClient['Id']
            ];

        } catch (\Exception $e) {
            $this->log('ERROR', "Failed to update company {$companyData['coName']}: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to update company: ' . $e->getMessage(),
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Validate company data from eProject
     */
    private function validateCompanyData(array $companyData): void
    {
        $requiredFields = ['coID', 'coName', 'firstName', 'lastName', 'email'];
        
        foreach ($requiredFields as $field) {
            if (empty($companyData[$field])) {
                throw new \InvalidArgumentException("Missing required field: {$field}");
            }
        }

        // Validate email format
        if (!filter_var($companyData['email'], FILTER_VALIDATE_EMAIL)) {
            throw new \InvalidArgumentException("Invalid email format: {$companyData['email']}");
        }
    }

    /**
     * Prepare client data for ShareFile API
     */
    private function prepareClientData(array $companyData): array
    {
        return [
            'companyName' => $companyData['coName'],
            'firstName' => $companyData['firstName'],
            'lastName' => $companyData['lastName'],
            'email' => $companyData['email'],
            'clientId' => $companyData['coID'],
            'internalDisplayName' => $companyData['coName'] . ' (' . $companyData['coID'] . ')'
        ];
    }

    /**
     * Create project folder from template
     */
    private function createProjectFolder(string $clientId, array $companyData): array
    {
        $folderName = $companyData['coName'] . ' - Project Files';
        
        return $this->shareFileAPI->createFolderFromTemplate(
            $clientId,
            $folderName,
            $_ENV['SHAREFILE_TEMPLATE_FOLDER_ID']
        );
    }
} 