<!--- 
Example Lucee Integration for eProject ShareFile Connector
This file shows how to integrate the ShareFile connector when a new company is saved in eProject.

Place this code in your company save/insert logic in eProject.
--->
<cfscript>
// Example: After successfully saving a new company in eProject
// This code should be placed in your company save logic

function integrateWithShareFile(companyData) {
    try {
        // Prepare the data for ShareFile API
        var shareFileData = {
            "coID": companyData.coID,
            "coName": companyData.coName,
            "firstName": companyData.primaryContact.firstName,
            "lastName": companyData.primaryContact.lastName,
            "email": companyData.primaryContact.email
        };

        // Call the ShareFile webhook
        var httpService = new http();
        httpService.setMethod("POST");
        httpService.setUrl("https://yourserver.com/path/to/sharefile-connector/webhook.php");
        httpService.addParam(type="header", name="Content-Type", value="application/json");
        httpService.addParam(type="body", value=serializeJSON(shareFileData));
        httpService.setTimeout(30); // 30 second timeout

        var result = httpService.send().getPrefix();
        
        if (result.status_code == 200) {
            var response = deserializeJSON(result.fileContent);
            
            if (response.success) {
                // Log successful integration
                writeLog(
                    file="sharefile_integration",
                    text="ShareFile integration successful for company: #companyData.coName# (coID: #companyData.coID#). ShareFile Client ID: #response.clientId#",
                    type="information"
                );
                
                // Optionally store ShareFile client ID in your database
                // updateCompanyShareFileInfo(companyData.coID, response.clientId, response.folderId);
                
                return {
                    success: true,
                    message: "ShareFile integration successful",
                    shareFileClientId: response.clientId,
                    shareFileFolderId: response.folderId
                };
            } else {
                // Log integration failure
                writeLog(
                    file="sharefile_integration",
                    text="ShareFile integration failed for company: #companyData.coName# (coID: #companyData.coID#). Error: #response.message#",
                    type="error"
                );
                
                return {
                    success: false,
                    message: "ShareFile integration failed: " & response.message
                };
            }
        } else {
            // Log HTTP error
            writeLog(
                file="sharefile_integration",
                text="ShareFile integration HTTP error for company: #companyData.coName# (coID: #companyData.coID#). Status: #result.status_code#, Response: #result.fileContent#",
                type="error"
            );
            
            return {
                success: false,
                message: "ShareFile integration HTTP error: " & result.status_code
            };
        }
        
    } catch (any e) {
        // Log exception
        writeLog(
            file="sharefile_integration",
            text="ShareFile integration exception for company: #companyData.coName# (coID: #companyData.coID#). Error: #e.message#",
            type="error"
        );
        
        return {
            success: false,
            message: "ShareFile integration error: " & e.message
        };
    }
}

// Example usage in your company save logic:
/*
<cfscript>
// After successfully saving the company to your database
var companyData = {
    coID: form.coID,
    coName: form.coName,
    primaryContact: {
        firstName: form.primaryContactFirstName,
        lastName: form.primaryContactLastName,
        email: form.primaryContactEmail
    }
};

// Integrate with ShareFile
var shareFileResult = integrateWithShareFile(companyData);

if (shareFileResult.success) {
    // Show success message to user
    application.message = "Company saved successfully. ShareFile integration completed.";
} else {
    // Show warning message to user (company was saved but ShareFile integration failed)
    application.warning = "Company saved successfully, but ShareFile integration failed: " & shareFileResult.message;
}
</cfscript>
*/

// Optional: Function to update company with ShareFile information
function updateCompanyShareFileInfo(coID, shareFileClientId, shareFileFolderId) {
    try {
        // Update your company table with ShareFile information
        var sql = "
            UPDATE Companies 
            SET shareFileClientId = :shareFileClientId,
                shareFileFolderId = :shareFileFolderId,
                shareFileIntegrationDate = GETDATE()
            WHERE coID = :coID
        ";
        
        var params = {
            shareFileClientId: shareFileClientId,
            shareFileFolderId: shareFileFolderId,
            coID: coID
        };
        
        queryExecute(sql, params);
        
    } catch (any e) {
        writeLog(
            file="sharefile_integration",
            text="Failed to update company ShareFile info for coID: #coID#. Error: #e.message#",
            type="error"
        );
    }
}

// Optional: Function to handle company updates
function updateShareFileClient(companyData) {
    try {
        // Prepare the data for ShareFile API update
        var shareFileData = {
            "coID": companyData.coID,
            "coName": companyData.coName,
            "firstName": companyData.primaryContact.firstName,
            "lastName": companyData.primaryContact.lastName,
            "email": companyData.primaryContact.email
        };

        // Call the ShareFile webhook with update flag
        var httpService = new http();
        httpService.setMethod("POST");
        httpService.setUrl("https://yourserver.com/path/to/sharefile-connector/webhook.php");
        httpService.addParam(type="header", name="Content-Type", value="application/json");
        httpService.addParam(type="header", name="X-Operation", value="update");
        httpService.addParam(type="body", value=serializeJSON(shareFileData));
        httpService.setTimeout(30);

        var result = httpService.send().getPrefix();
        
        if (result.status_code == 200) {
            var response = deserializeJSON(result.fileContent);
            
            if (response.success) {
                writeLog(
                    file="sharefile_integration",
                    text="ShareFile client update successful for company: #companyData.coName# (coID: #companyData.coID#)",
                    type="information"
                );
                
                return {
                    success: true,
                    message: "ShareFile client updated successfully"
                };
            } else {
                writeLog(
                    file="sharefile_integration",
                    text="ShareFile client update failed for company: #companyData.coName# (coID: #companyData.coID#). Error: #response.message#",
                    type="error"
                );
                
                return {
                    success: false,
                    message: "ShareFile client update failed: " & response.message
                };
            }
        } else {
            writeLog(
                file="sharefile_integration",
                text="ShareFile client update HTTP error for company: #companyData.coName# (coID: #companyData.coID#). Status: #result.status_code#",
                type="error"
            );
            
            return {
                success: false,
                message: "ShareFile client update HTTP error: " & result.status_code
            };
        }
        
    } catch (any e) {
        writeLog(
            file="sharefile_integration",
            text="ShareFile client update exception for company: #companyData.coName# (coID: #companyData.coID#). Error: #e.message#",
            type="error"
        );
        
        return {
            success: false,
            message: "ShareFile client update error: " & e.message
        };
    }
}
</cfscript>

<!--- 
Example HTML form integration (if you want to show integration status to users)
--->
<cfform name="companyForm" action="saveCompany.cfm" method="post">
    <!--- Your existing company form fields --->
    
    <!--- Display ShareFile integration status if available --->
    <cfif structKeyExists(session, "shareFileIntegrationStatus")>
        <div class="alert alert-info">
            <strong>ShareFile Integration:</strong> #session.shareFileIntegrationStatus#
        </div>
        <cfset structDelete(session, "shareFileIntegrationStatus")>
    </cfif>
    
    <!--- Your form submit button --->
    <input type="submit" value="Save Company" class="btn btn-primary">
</cfform> 