# eProject ShareFile Connector

A PHP connector to automatically integrate eProject with ShareFile API. This connector creates ShareFile clients and project folders when new companies are created in eProject.

## Features

- **Automatic Client Creation**: Creates ShareFile clients when new companies are added to eProject
- **Template Folder Creation**: Creates project folders from ShareFile templates
- **Webhook Integration**: Supports webhook triggers from Lucee
- **Database Integration**: Supports direct database triggers
- **Error Handling**: Comprehensive error handling and logging
- **Update Support**: Supports updating existing client information (Phase 2)

## Requirements

- PHP 7.4 or higher
- Composer
- ShareFile API credentials (OAuth2)
- Access to eProject database (for database trigger method)

## Installation

1. **Clone or download this repository** to your server
2. **Install dependencies**:
   ```bash
   composer install
   ```
3. **Copy environment configuration**:
   ```bash
   cp env.example .env
   ```
4. **Configure your environment variables** in the `.env` file

## Configuration

Edit the `.env` file with your ShareFile API credentials and settings:

```env
# ShareFile API Configuration
SHAREFILE_CLIENT_ID=your_client_id_here
SHAREFILE_CLIENT_SECRET=your_client_secret_here
SHAREFILE_USERNAME=your_username_here
SHAREFILE_PASSWORD=your_password_here
SHAREFILE_SUBDOMAIN=your_subdomain_here

# Database Configuration (for direct database triggers)
DB_HOST=localhost
DB_NAME=eproject_db
DB_USER=db_username
DB_PASSWORD=db_password

# Logging Configuration
LOG_LEVEL=debug
LOG_FILE=logs/sharefile_connector.log

# Template Configuration
SHAREFILE_TEMPLATE_FOLDER_ID=your_template_folder_id_here
```

### Getting ShareFile API Credentials

1. Log into your ShareFile account
2. Go to Admin → API
3. Create a new API application
4. Note down the Client ID and Client Secret
5. Your subdomain is the part before `.sharefile.com` in your ShareFile URL

### Finding Template Folder ID

1. In ShareFile, navigate to the template folder you want to use
2. The folder ID is in the URL: `https://yourdomain.sharefile.com/folder/ID`
3. Copy the ID and set it in `SHAREFILE_TEMPLATE_FOLDER_ID`

## Usage

### Method 1: Webhook Integration (Recommended)

This method integrates with Lucee to trigger the connector when a new company is saved.

#### Lucee Integration

Add this code to your Lucee application when a new company is saved:

```cfml
<cfscript>
// After successfully saving the company
companyData = {
    "coID": company.coID,
    "coName": company.coName,
    "firstName": company.primaryContact.firstName,
    "lastName": company.primaryContact.lastName,
    "email": company.primaryContact.email
};

// Call the webhook
httpService = new http();
httpService.setMethod("POST");
httpService.setUrl("https://yourserver.com/path/to/sharefile-connector/webhook.php");
httpService.addParam(type="header", name="Content-Type", value="application/json");
httpService.addParam(type="body", value=serializeJSON(companyData));

try {
    result = httpService.send().getPrefix();
    if (result.status_code == 200) {
        writeOutput("ShareFile integration successful");
    } else {
        writeOutput("ShareFile integration failed: " & result.fileContent);
    }
} catch (any e) {
    writeOutput("ShareFile integration error: " & e.message);
}
</cfscript>
```

#### Testing the Webhook

You can test the webhook with curl:

```bash
curl -X POST https://yourserver.com/path/to/sharefile-connector/webhook.php \
  -H "Content-Type: application/json" \
  -d '{
    "coID": "12345",
    "coName": "Test Company",
    "firstName": "John",
    "lastName": "Doe",
    "email": "john.doe@testcompany.com"
  }'
```

### Method 2: Database Trigger

This method uses a database trigger to automatically call the connector.

#### Database Trigger Setup

Create a trigger in your SQL Server database:

```sql
CREATE TRIGGER tr_Company_Insert_ShareFile
ON Companies
AFTER INSERT
AS
BEGIN
    DECLARE @coID VARCHAR(50)
    SELECT @coID = coID FROM inserted
    
    -- Call the PHP script
    EXEC xp_cmdshell 'php /path/to/sharefile-connector/database_trigger.php ' + @coID
END
```

#### Testing the Database Trigger

```bash
php database_trigger.php 12345
```

## Data Mapping

The connector maps eProject data to ShareFile as follows:

| eProject Field | ShareFile Field | Description |
|----------------|-----------------|-------------|
| coID | ClientID | Unique identifier for the client |
| coName | CompanyName | Company name |
| firstName | FirstName | Primary contact first name |
| lastName | LastName | Primary contact last name |
| email | Email | Primary contact email |
| coName + (coID) | InternalDisplayName | Display name in ShareFile |

## Error Handling

The connector includes comprehensive error handling:

- **Validation Errors**: Missing required fields, invalid email format
- **API Errors**: ShareFile API communication issues
- **Duplicate Clients**: Handles cases where clients already exist
- **Logging**: All operations are logged for debugging

### Common Error Scenarios

1. **Client Already Exists**: The connector will log a warning and return the existing client ID
2. **Invalid API Credentials**: Check your ShareFile API credentials in the `.env` file
3. **Network Issues**: The connector will retry and log the error
4. **Missing Template**: If no template folder ID is configured, only the client will be created

## Logging

Logs are written to the file specified in `LOG_FILE` environment variable. Log levels:

- **DEBUG**: Detailed debugging information
- **INFO**: General information about operations
- **WARNING**: Non-critical issues (e.g., client already exists)
- **ERROR**: Critical errors that prevent operation

## Security Considerations

1. **Environment Variables**: Never commit your `.env` file to version control
2. **API Credentials**: Store ShareFile credentials securely
3. **Webhook Security**: Consider adding authentication to the webhook endpoint
4. **Database Access**: Use minimal database permissions for the connector

## Troubleshooting

### Common Issues

1. **Authentication Failed**
   - Verify ShareFile API credentials
   - Check subdomain configuration
   - Ensure API application has correct permissions

2. **Client Creation Fails**
   - Verify all required fields are provided
   - Check ShareFile API documentation for field requirements
   - Review logs for specific error messages

3. **Template Folder Not Created**
   - Verify `SHAREFILE_TEMPLATE_FOLDER_ID` is set correctly
   - Ensure template folder exists and is accessible
   - Check ShareFile permissions for template access

### Debug Mode

Set `LOG_LEVEL=debug` in your `.env` file for detailed logging information.

## Phase 2 Features

Future enhancements planned:

- **Client Updates**: Support for updating existing client information
- **User Management**: Add/remove users from ShareFile clients
- **Folder Permissions**: Manage folder access permissions
- **Bulk Operations**: Process multiple companies at once
- **Web Interface**: Admin interface for managing integrations

## Support

For issues and questions:

1. Check the logs for error details
2. Verify your configuration settings
3. Test with the provided examples
4. Review ShareFile API documentation at api.sharefile.com/docs

## License

This project is provided as-is for internal use. Please ensure compliance with ShareFile API terms of service. 