# External PHP Server Setup Guide

This guide will help you deploy the ShareFile connector on a separate PHP server and integrate it with your Lucee server.

## Architecture Overview

```
┌─────────────────┐    HTTP Request    ┌──────────────────┐    API Call    ┌─────────────┐
│   Lucee Server  │ ──────────────────► │  PHP Server      │ ──────────────► │  ShareFile  │
│   (eProject)    │                    │  (Connector)     │                │     API     │
└─────────────────┘                    └──────────────────┘                └─────────────┘
```

## Step 1: Deploy to PHP Server

### Files to Upload

Upload these files to your PHP server:

```
sharefile-connector/
├── simple_webhook.php          # Main webhook endpoint
├── test_simple.php             # Test script
├── .env                        # Configuration (create this)
├── src/
│   ├── SimpleShareFileAPI.php  # API client
│   └── ClientManager.php       # Business logic
└── logs/                       # Log directory (will be created)
```

### Server Requirements

- **PHP 7.4+** with cURL extension
- **Web server** (Apache/Nginx)
- **SSL certificate** (recommended for production)
- **Internet access** to ShareFile API

## Step 2: Configure the PHP Server

### 1. Create .env File

Create a `.env` file on your PHP server:

```env
# ShareFile API Configuration
SHAREFILE_CLIENT_ID=your_client_id_here
SHAREFILE_CLIENT_SECRET=your_client_secret_here
SHAREFILE_USERNAME=your_username_here
SHAREFILE_PASSWORD=your_password_here
SHAREFILE_SUBDOMAIN=your_subdomain_here

# Logging Configuration
LOG_FILE=logs/sharefile_connector.log

# Template Configuration (optional)
SHAREFILE_TEMPLATE_FOLDER_ID=your_template_folder_id_here
```

### 2. Set Permissions

```bash
# Make logs directory writable
chmod 755 logs/
chmod 644 .env
```

### 3. Test the PHP Server

SSH into your PHP server and run:

```bash
cd /path/to/sharefile-connector
php test_simple.php
```

You should see output like:
```
ShareFile Connector Test
=======================

Checking environment variables...
✓ SHAREFILE_CLIENT_ID is set
✓ SHAREFILE_CLIENT_SECRET is set
...

✓ SUCCESS: Company created in ShareFile
  ShareFile Client ID: abc123
```

## Step 3: Configure Lucee Server

### 1. Update Your Lucee Code

Use the external integration example (`example_lucee_external_integration.cfm`) in your eProject application.

### 2. Set PHP Server URL

Replace `https://your-php-server.com/sharefile-connector` with your actual PHP server URL.

### 3. Test the Integration

Add this test code to your Lucee application:

```cfml
<cfscript>
// Test the external PHP server
var testData = {
    "coID": "TEST" & createUUID(),
    "coName": "Test Company",
    "firstName": "John",
    "lastName": "Doe",
    "email": "john.doe@testcompany.com"
};

var phpServerUrl = "https://your-php-server.com/sharefile-connector";
var result = integrateWithShareFileExternal(testData, phpServerUrl);

writeOutput("Integration result: " & serializeJSON(result));
</cfscript>
```

## Step 4: Security Considerations

### 1. HTTPS Only

Always use HTTPS for communication between servers:

```cfml
var phpServerUrl = "https://your-php-server.com/sharefile-connector";
```

### 2. API Authentication (Optional)

Add authentication to your PHP webhook:

```php
// In simple_webhook.php
$apiKey = $_SERVER['HTTP_X_API_KEY'] ?? '';
if ($apiKey !== $_ENV['WEBHOOK_API_KEY']) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}
```

Then in Lucee:
```cfml
httpService.addParam(type="header", name="X-API-Key", value="your_api_key_here");
```

### 3. IP Whitelisting

Restrict access to your PHP webhook by IP address in your web server configuration.

## Step 5: Monitoring and Logging

### 1. PHP Server Logs

Monitor the PHP server logs:
```bash
tail -f /path/to/sharefile-connector/logs/sharefile_connector.log
```

### 2. Lucee Server Logs

Monitor Lucee integration logs:
```cfml
// Check Lucee logs
writeLog(file="sharefile_integration", text="Integration status", type="information");
```

### 3. Error Handling

The integration includes comprehensive error handling:
- Network timeouts
- API errors
- Invalid data
- Duplicate clients

## Step 6: Production Deployment

### 1. Environment Variables

Use proper environment variable management on your PHP server:
- **Development**: `.env` file
- **Production**: Server environment variables

### 2. Log Rotation

Set up log rotation for the PHP server:
```bash
# /etc/logrotate.d/sharefile-connector
/path/to/sharefile-connector/logs/*.log {
    daily
    rotate 30
    compress
    delaycompress
    missingok
    notifempty
}
```

### 3. Monitoring

Set up monitoring for:
- PHP server availability
- ShareFile API connectivity
- Integration success rates

## Troubleshooting

### Common Issues

1. **"Connection refused"**
   - Check PHP server URL
   - Verify PHP server is running
   - Check firewall settings

2. **"Timeout"**
   - Increase timeout in Lucee HTTP call
   - Check PHP server performance
   - Verify ShareFile API response times

3. **"Authentication failed"**
   - Verify ShareFile credentials on PHP server
   - Check `.env` file permissions
   - Test with `test_simple.php`

4. **"Invalid JSON"**
   - Check data format in Lucee
   - Verify Content-Type header
   - Test with curl

### Testing Commands

Test the PHP webhook directly:
```bash
curl -X POST https://your-php-server.com/sharefile-connector/simple_webhook.php \
  -H "Content-Type: application/json" \
  -d '{
    "coID": "TEST123",
    "coName": "Test Company",
    "firstName": "John",
    "lastName": "Doe",
    "email": "john.doe@testcompany.com"
  }'
```

## Benefits of External Setup

✅ **No PHP required** on Lucee server
✅ **Clean separation** of technologies
✅ **Easier maintenance** and updates
✅ **Better security** (API credentials isolated)
✅ **Scalability** (can run multiple instances)
✅ **Independent logging** and monitoring

## Next Steps

1. **Deploy to PHP server** and test
2. **Configure Lucee integration** with external URL
3. **Set up monitoring** and logging
4. **Test with real data** from eProject
5. **Go live** with production integration 